<?php
// app/Livewire/PublicRegistrationForm.php

namespace App\Livewire;

use App\Models\Kabupaten;
use App\Models\Kegiatan;
use App\Models\Pendaftaran;
use App\Models\Peserta;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Illuminate\Support\Facades\DB;
use Livewire\Component;

class PublicRegistrationForm extends Component implements HasForms
{
    use InteractsWithForms;

    public array $kabupatenOptions = [];

    public ?array $data = [];

    public function mount(): void
    {
        $this->kabupatenOptions = Kabupaten::pluck('nama', 'id')->toArray();
        $this->form->fill();
    }

    // public function checkNikUnique($nik): void
    // {
    //     if(strlen($nik) === 16) {
    //         $exists = Peserta::where('nik', $nik)->exists();
    //         if ($exists) {
    //             $this->addError('data.nik', 'NIK ini telah terdaftar.');
    //         } else {
    //             $this->resetErrorBag('data.nik');
    //         }
    //     } else {
    //         $this->resetErrorBag('data.nik');
    //     }
    // }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                // ... (kode form yang sudah diperbaiki sebelumnya tetap sama)
                // Section 1: Data Personal
                \Filament\Forms\Components\Section::make('Data Personal')
                    ->schema([
                        Select::make('kabupaten_id')
                            ->live()
                            ->label('Kabupaten')
                            ->options($this->kabupatenOptions)
                            ->required(),
                            // ->searchable(),
                        TextInput::make('nik')
                            ->label('NIK')
                            ->required()
                            ->length(16)
                            ->unique('pesertas', 'nik')
                            // ->reactive()
                            // ->afterStateUpdated(fn($state) => $this->checkNikUnique($state))
                            ->validationMessages([
                                'unique' => 'NIK ini sudah terdaftar. Silakan gunakan NIK lain atau hubungi admin.',
                            ]),
                        TextInput::make('nama')
                            ->label('Nama Lengkap')
                            ->required()
                            ->maxLength(255)
                            ->afterStateUpdated(function ($state, $set) {
                                $set('nama', strtoupper($state));
                            }),
                        TextInput::make('tempat_lahir')
                            ->label('Tempat Lahir')
                            ->required(),
                        DatePicker::make('tanggal_lahir')
                            ->label('Tanggal Lahir')
                            ->required()
                            ->maxDate(now()->subYears(18))
                            ->validationMessages([
                                'max' => 'Usia minimal 18 tahun.',
                            ]),
                        TextInput::make('nomor_telepon')
                            ->label('Nomor Telepon')
                            ->prefix("'")
                            ->required()
                            ->tel()
                            ->maxLength(15),
                    ])->columns(2),

                // ... (section lainnya tetap sama)
                // Section 2: Data Tambahan
                \Filament\Forms\Components\Section::make('Data Tambahan')
                    ->schema([
                        Select::make('agama')
                            ->label('Agama')
                            ->options([
                                'ISLAM' => 'ISLAM',
                                'KRISTEN' => 'KRISTEN',
                                'KATOLIK' => 'KATOLIK',
                                'HINDU' => 'HINDU',
                                'BUDDHA' => 'BUDDHA',
                                'KONGHUCU' => 'KONGHUCU'
                            ])
                            ->required(),
                        Select::make('jenis_kelamin')
                            ->label('Jenis Kelamin')
                            ->options([
                                'LAKI-LAKI' => 'LAKI-LAKI',
                                'PEREMPUAN' => 'PEREMPUAN'
                            ])
                            ->required(),
                        Select::make('status_pernikahan')
                            ->label('Status Pernikahan')
                            ->options([
                                'BELUM MENIKAH' => 'BELUM MENIKAH',
                                'MENIKAH' => 'MENIKAH',
                                'CERAI HIDUP' => 'CERAI HIDUP',
                                'CERAI MATI' => 'CERAI MATI'
                            ])
                            ->required(),
                        Select::make('pendidikan_terakhir')
                            ->label('Pendidikan Terakhir')
                            ->options([
                                'SD' => 'SD', 'SMP' => 'SMP', 'SMA' => 'SMA',
                                'D1' => 'D1', 'D2' => 'D2', 'D3' => 'D3',
                                'S1' => 'S1', 'S2' => 'S2', 'S3' => 'S3'
                            ])
                            ->required(),
                        TextInput::make('pekerjaan')
                            ->label('Pekerjaan')
                            ->required(),
                        TextInput::make('usaha_tani')
                            ->label('Usaha Tani')
                            ->required(),
                    ])->columns(3),

                // Section 3: Alamat dan Kontak
                \Filament\Forms\Components\Section::make('Alamat dan Kontak')
                    ->schema([
                        Textarea::make('alamat_lengkap')
                            ->label('Alamat Lengkap (Sesuai Identitas Kependudukan)')
                            ->required()
                            ->placeholder('Contoh: RT.001 RW.002 DS. GEDAWANG, KEC. SEMAMPIR, KAB. TEMANGGUNG')
                            ->columnSpanFull(),
                        TextInput::make('nama_poktan')
                            ->label('Nama Poktan/Organisasi/Departemen')
                            ->required(),
                        Textarea::make('alamat_poktan')
                            ->label('Alamat Poktan/Organisasi/Departemen')
                            ->required()
                            ->columnSpanFull(),
                        TextInput::make('nip')
                            ->label('NIP (Opsional bagi Aparatur)')
                            ->nullable(),
                        TextInput::make('email')
                            ->label('Alamat Email')
                            ->email()
                            ->required(),
                    ])->columns(2),

                // Section 4: Pilihan Pelatihan
                \Filament\Forms\Components\Section::make('Pilihan Pelatihan')
                    ->schema([
                        Select::make('kegiatan_id')
                            ->label('Pilih Jenis Pelatihan')
                            ->options(
                                Kegiatan::aktif()
                                    ->get()
                                    ->filter(fn($kegiatan) => $kegiatan->kuota_tersedia)
                                    ->mapWithKeys(fn ($kegiatan) => [
                                        $kegiatan->id => "{$kegiatan->kode_pelatihan} - {$kegiatan->nama_pelatihan} (Kuota: {$kegiatan->jumlah_peserta_diterima}/{$kegiatan->kuota}) - {$kegiatan->tanggal_mulai->format('d M Y')} s/d {$kegiatan->tanggal_selesai->format('d M Y')}"
                                    ])
                            )
                            ->required()
                            // ->searchable()
                            ->validationMessages([
                                'required' => 'Silakan pilih salah satu jenis pelatihan.',
                            ]),
                    ]),
            ])
            ->statePath('data');
    }

    public function create(): void
    {
        try {
            DB::transaction(function () {
                $data = $this->form->getState();

                // Ensure nama is uppercase
                $data['nama'] = strtoupper($data['nama']);

                // Cek usia maksimal 50 tahun
                $usia = now()->diffInYears($data['tanggal_lahir']);
                if ($usia > 50) {
                    throw new \Exception('Usia peserta melebihi batas maksimal 50 tahun.');
                }

                // Cek apakah NIK sudah pernah mendaftar di kegiatan yang sama
                $existingRegistration = Pendaftaran::where('peserta_nik', $data['nik'])
                    ->where('kegiatan_id', $data['kegiatan_id'])
                    ->exists();

                if ($existingRegistration) {
                    throw new \Exception('Anda sudah terdaftar pada pelatihan ini sebelumnya.');
                }

                // Cek kuota kegiatan
                $kegiatan = Kegiatan::find($data['kegiatan_id']);
                $pesertaDiterima = $kegiatan->pendaftarans()
                    ->where('status', 'diterima')
                    ->count();
                if ($pesertaDiterima >= $kegiatan->kuota) {
                    throw new \Exception('Kuota untuk pelatihan ini sudah penuh.');
                }

                // Simpan peserta
                $peserta = Peserta::create($data);

                // Simpan pendaftaran
                Pendaftaran::create([
                    'peserta_nik' => $peserta->nik,
                    'kegiatan_id' => $data['kegiatan_id'],
                    'status' => 'pending',
                ]);

                session()->flash('success', 'Pendaftaran berhasil! Status pendaftaran Anda akan diverifikasi oleh admin.');
                $this->form->fill();
            });
        } catch (\Exception $e) {
            session()->flash('error', $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.public-registration-form')
            ->layout('livewire.layouts.public');
    }
}